﻿// IMPORTANT: Filter name and keywords filename must match

uses
  Classes, Common, DataEntry, DataStorage, Regex, SysUtils;

// ------------------------------------------------------------------------------
// Function: Test Valid Regex
// ------------------------------------------------------------------------------
function IsValidRegEx(aStr: string): boolean;
var
  aRegEx: TRegEx;
begin
  Result := False;
  aRegEx := TRegEx.Create;
  try
    aRegEx.SearchTerm := aStr;
    Result := aRegEx.LastError = 0;
  finally
    aRegEx.free;
  end;
end;

// ------------------------------------------------------------------------------
// Function: Remove Last Part
// ------------------------------------------------------------------------------
function RemoveLastPart(var s: string): string;
begin
  repeat
    delete(s, length(s), 1)
  until (s[length(s)] = '\') or (s = '');
  Result := s;
end;

// ------------------------------------------------------------------------------
// On Start: Read the file from disk and create the regex string
// ------------------------------------------------------------------------------
procedure OnStartFilter(aDataStore: TDataStore; Var DoContinue: boolean; Var OutObj: string);
var
  aKeywords_StringList: TStringList;
  FilterLogging_StringList: TStringList;
  CommentString: string;
  i: integer;
  OneBigRegex: string;
  FilterPath_str: string;
  LogFileFullPath_str: string;
  FilterFileName_pas_str: string;
  FilterFileName_txt_str: string;
  FilterFileName_log_str: string;

begin
  DoContinue := True;
  FilterLogging_StringList := TStringList.Create;
  aKeywords_StringList := TStringList.Create;
  aKeywords_StringList.Delimiter := '|';
  aKeywords_StringList.StrictDelimiter := True;

  // -----------------------------------------------------------
  // The filter knows where it is. Get the path and name
  // -----------------------------------------------------------
  FilterPath_str := CmdLine.Path;
  FilterFileName_pas_str := ExtractFileName(FilterPath_str);

  if RegexMatch(FilterPath_str, '\\filters', False) then
  begin

    // Remove the path back to the point where there is no filters folder
    while RegexMatch(FilterPath_str, '\\filters', False) do
      FilterPath_str := RemoveLastPart(FilterPath_str);

    // Step forward into the keywords folder and re-add the script name
    if DirectoryExists(FilterPath_str) then
    begin
      FilterPath_str := FilterPath_str + 'keywords\regex\' + FilterFileName_pas_str;
      FilterFileName_txt_str := StringReplace(FilterPath_str, '.pas', '.txt', [rfReplaceAll, rfIgnoreCase]);
    end;

  end;

  // -----------------------------------------------------------
  // Once the Keyword Files is Found
  // -----------------------------------------------------------
  try
    if FileExists(FilterFileName_txt_str) then
    begin
      aKeywords_StringList.LoadFromFile(FilterFileName_txt_str);

      FilterLogging_StringList.Add('Found keywords file: ' + FilterFileName_txt_str);
      FilterLogging_StringList.Add('Contains keywords: ' + IntToStr(aKeywords_StringList.Count));

      // Remove blank lines from StringList
      for i := aKeywords_StringList.Count - 1 downto 0 do
      begin
        aKeywords_StringList[i] := TrimRight(aKeywords_StringList[i]);
        if Trim(aKeywords_StringList[i]) = '' then
          aKeywords_StringList.delete(i);
      end;

      // Remove comment lines from StringList
      for i := aKeywords_StringList.Count - 1 downto 0 do
      begin
        CommentString := copy(aKeywords_StringList(i), 1, 1);
        if CommentString = '#' then
          aKeywords_StringList.delete(i);
      end;

      for i := 0 to aKeywords_StringList.Count - 1 do
      begin
        FilterLogging_StringList.Add('Keyword ' + IntToStr(i) + ': ' + aKeywords_StringList(i));
      end;

    end
    else
    begin
      FilterLogging_StringList.Add('Did not find: ' + FilterFileName_txt_str);
    end;

    // Convert the keywords into one big regex statement
    OneBigRegex := aKeywords_StringList.DelimitedText;

    FilterLogging_StringList.Add('OneBigRegex: ' + OneBigRegex);

    // Test the regex
    if IsValidRegEx(OneBigRegex) then
      FilterLogging_StringList.Add('Valid regex: ' + OneBigRegex)
    else
      FilterLogging_StringList.Add('Invalid regex: ' + OneBigRegex);

    OutObj := OneBigRegex;
    FilterLogging_StringList.Add('OutObj: ' + OutObj);

    // Save the manual log file
    FilterFileName_log_str := StringReplace(FilterFileName_pas_str, '.pas', '.log', [rfReplaceAll, rfIgnoreCase]);
    LogFileFullPath_str := GetCurrentCaseDir + 'Logs' + '\' + UpperCase(FilterFileName_log_str);
    FilterLogging_StringList.Add('Log File: ' + LogFileFullPath_str);
    FilterLogging_StringList.SaveToFile(LogFileFullPath_str);

  finally
    aKeywords_StringList.free;
    FilterLogging_StringList.free;
  end;
end;

// ------------------------------------------------------------------------------
// Run the filter
// ------------------------------------------------------------------------------
function FilterEntry(anEntry: TEntry; InObj: string): integer;
begin
  Result := -1; // Exclude
  if assigned(anEntry) then
  begin
    if RegexMatch(anEntry.EntryName, InObj, False) then
    begin
      Result := 1; // Include
    end;
  end;
end;

begin

end.
